/*
 * Copyright © 2016 Red Hat.
 * Copyright © 2016 Bas Nieuwenhuizen
 *
 * based in part on anv driver which is:
 * Copyright © 2015 Intel Corporation
 *
 * SPDX-License-Identifier: MIT
 */

#ifndef RADV_PHYSICAL_DEVICE_H
#define RADV_PHYSICAL_DEVICE_H

#include "ac_gpu_info.h"
#include "ac_perfcounter.h"

#include "ac_uvd_dec.h"
#include "ac_vcn_enc.h"
#include "radv_constants.h"
#include "radv_instance.h"
#include "radv_queue.h"
#include "radv_radeon_winsys.h"
#include "wsi_common.h"

#include "compiler/shader_enums.h"
#include "nir_shader_compiler_options.h"

#include "vk_physical_device.h"

#ifndef _WIN32
#include <xf86drm.h>
#endif

struct radv_binning_settings {
   unsigned context_states_per_bin;    /* allowed range: [1, 6] */
   unsigned persistent_states_per_bin; /* allowed range: [1, 32] */
   unsigned fpovs_per_batch;           /* allowed range: [0, 255], 0 = unlimited */
};

struct radv_physical_device_cache_key {
   enum radeon_family family;
   uint32_t ptr_size;

   uint32_t conformant_trunc_coord : 1;
   uint32_t clear_lds : 1;
   uint32_t cs_wave32 : 1;
   uint32_t disable_aniso_single_level : 1;
   uint32_t disable_shrink_image_store : 1;
   uint32_t disable_sinking_load_input_fs : 1;
   uint32_t disable_trunc_coord : 1;
   uint32_t emulate_rt : 1;
   uint32_t bvh8 : 1;
   uint32_t ge_wave32 : 1;
   uint32_t invariant_geom : 1;
   uint32_t no_fmask : 1;
   uint32_t no_ngg_gs : 1;
   uint32_t no_rt : 1;
   uint32_t ps_wave32 : 1;
   uint32_t rt_wave64 : 1;
   uint32_t split_fma : 1;
   uint32_t ssbo_non_uniform : 1;
   uint32_t tex_non_uniform : 1;
   uint32_t lower_terminate_to_discard : 1;
   uint32_t use_llvm : 1;
   uint32_t use_ngg : 1;
   uint32_t use_ngg_culling : 1;
   uint32_t no_implicit_varying_subgroup_size : 1;
   uint32_t mitigate_smem_oob : 1;

   uint32_t reserved : 8;
};

enum radv_video_enc_hw_ver {
   RADV_VIDEO_ENC_HW_1_2,
   RADV_VIDEO_ENC_HW_2,
   RADV_VIDEO_ENC_HW_3,
   RADV_VIDEO_ENC_HW_4,
   RADV_VIDEO_ENC_HW_5,
};

/**
 * Description of the various HiZ workarounds for GFX12.
 *
 * - disabled: None of the HiZ/HiS workarounds are enabled. This is very risky and should only be
 *   used when we guarantee no issues. Performance is optimal.
 *
 * - partial: Emit BOTTOM_OF_PIPE_TS events after every draw to mitigate the issue. This is
 *   potentially risky because it doesn't mitigate the issue complety but it helps in most cases.
 *   Performance should be mostly optimal.
 *
 * - full: Disable HiZ/HiS at draw time when required to prevent the issue to happen. This solution
 *   should be completely safe but it might decrease performance in some cases.
 */
enum radv_gfx12_hiz_wa {
   RADV_GFX12_HIZ_WA_DISABLED,
   RADV_GFX12_HIZ_WA_PARTIAL,
   RADV_GFX12_HIZ_WA_FULL,
};

struct radv_physical_device {
   struct vk_physical_device vk;

   struct radeon_winsys *ws;
   struct radeon_info info;
   char name[VK_MAX_PHYSICAL_DEVICE_NAME_SIZE];
   char marketing_name[VK_MAX_PHYSICAL_DEVICE_NAME_SIZE];
   uint8_t driver_uuid[VK_UUID_SIZE];
   uint8_t device_uuid[VK_UUID_SIZE];
   uint8_t cache_uuid[VK_UUID_SIZE];

   struct disk_cache *disk_cache_meta;

   struct ac_addrlib *addrlib;

   int local_fd;
   int master_fd;
   struct wsi_device wsi_device;

   /* Whether DCC should be enabled for MSAA textures. */
   bool dcc_msaa_allowed;

   /* Whether to enable FMASK compression for MSAA textures (GFX6-GFX10.3) */
   bool use_fmask;

   /* Whether to enable HTILE compression for depth/stencil images. */
   bool use_hiz;

   /* GFX12 HiZ workaround behavior. */
   enum radv_gfx12_hiz_wa gfx12_hiz_wa;

   /* Whether to enable NGG. */
   bool use_ngg;

   /* Whether to enable NGG culling. */
   bool use_ngg_culling;

   /* Whether to enable NGG streamout. */
   bool use_ngg_streamout;

   /* Whether to emulate the number of primitives generated by GS. */
   bool emulate_ngg_gs_query_pipeline_stat;

   /* Whether to emulate mesh/task shader queries. */
   bool emulate_mesh_shader_queries;

   /* Number of threads per wave. */
   uint8_t ps_wave_size;
   uint8_t cs_wave_size;
   uint8_t ge_wave_size;
   uint8_t rt_wave_size;

   /* Maximum compute shared memory size. */
   uint32_t max_shared_size;

   /* Whether to use the LLVM compiler backend */
   bool use_llvm;

   /* Whether to emulate ETC2 image support on HW without support. */
   bool emulate_etc2;

   /* Whether to emulate ASTC image support on HW without support. */
   bool emulate_astc;

   VkPhysicalDeviceMemoryProperties memory_properties;
   enum radeon_bo_domain memory_domains[VK_MAX_MEMORY_TYPES];
   enum radeon_bo_flag memory_flags[VK_MAX_MEMORY_TYPES];
   unsigned heaps;

   /* Bitmask of memory types that use the 32-bit address space. */
   uint32_t memory_types_32bit;

   /* Bitmask of memory types that are host-visible. */
   uint32_t memory_types_host_visible;

#ifndef _WIN32
   int available_nodes;
   drmPciBusInfo bus_info;

   dev_t primary_devid;
   dev_t render_devid;
#endif

   nir_shader_compiler_options nir_options[MESA_VULKAN_SHADER_STAGES];

   enum radv_queue_family vk_queue_to_radv[RADV_MAX_QUEUE_FAMILIES];
   uint32_t num_queues;

   uint32_t gs_table_depth;

   struct ac_task_info task_info;

   struct radv_binning_settings binning_settings;

   /* Performance counters. */
   struct ac_perfcounters ac_perfcounters;

   uint32_t num_perfcounters;
   struct radv_perfcounter_desc *perfcounters;

   struct {
      unsigned data0;
      unsigned data1;
      unsigned data2;
      unsigned cmd;
      unsigned cntl;
   } vid_dec_reg;
   enum amd_ip_type vid_decode_ip;
   uint32_t vid_addr_gfx_mode;
   struct ac_uvd_stream_handle stream_handle;
   uint32_t av1_version;
   rvcn_enc_cmd_t vcn_enc_cmds;
   enum radv_video_enc_hw_ver enc_hw_ver;
   uint32_t encoder_interface_version;
   bool video_encode_enabled;
   bool video_decode_enabled;
   struct radv_physical_device_cache_key cache_key;

   uint32_t tess_distribution_mode;

   struct {
      struct {
         uint32_t width;
         uint32_t height;
         uint32_t depth;
      } max_dims;

      uint32_t max_array_layers;
   } image_props;
};

VK_DEFINE_HANDLE_CASTS(radv_physical_device, vk.base, VkPhysicalDevice, VK_OBJECT_TYPE_PHYSICAL_DEVICE)

bool radv_sparse_enabled(const struct radv_physical_device *pdev);

static inline struct radv_instance *
radv_physical_device_instance(const struct radv_physical_device *pdev)
{
   return (struct radv_instance *)pdev->vk.instance;
}

static inline bool
radv_dedicated_sparse_queue_enabled(const struct radv_physical_device *pdev)
{
   /* Dedicated sparse queue requires VK_QUEUE_SUBMIT_MODE_THREADED, which is incompatible with
    * VK_DEVICE_TIMELINE_MODE_EMULATED. */
   return pdev->info.has_timeline_syncobj &&
          radv_sparse_enabled(pdev);
}

static inline bool
radv_has_shader_buffer_float_minmax(const struct radv_physical_device *pdev, unsigned bitsize)
{
   return (pdev->info.gfx_level <= GFX7 && !pdev->use_llvm) || pdev->info.gfx_level == GFX10 ||
          pdev->info.gfx_level == GFX10_3 || (pdev->info.gfx_level >= GFX11 && bitsize == 32);
}

static inline bool
radv_has_pops(const struct radv_physical_device *pdev)
{
   return pdev->info.gfx_level >= GFX9 && !pdev->use_llvm;
}

static inline bool
radv_has_uvd(struct radv_physical_device *pdev)
{
   enum radeon_family family = pdev->info.family;
   /* Only support UVD on TONGA+ */
   if (family < CHIP_TONGA)
      return false;
   return pdev->info.ip[AMD_IP_UVD].num_queues > 0;
}

static inline enum radv_queue_family
vk_queue_to_radv(const struct radv_physical_device *pdev, int queue_family_index)
{
   if (queue_family_index == VK_QUEUE_FAMILY_EXTERNAL || queue_family_index == VK_QUEUE_FAMILY_FOREIGN_EXT)
      return RADV_QUEUE_FOREIGN;
   if (queue_family_index == VK_QUEUE_FAMILY_IGNORED)
      return RADV_QUEUE_IGNORED;

   assert(queue_family_index < RADV_MAX_QUEUE_FAMILIES);
   return pdev->vk_queue_to_radv[queue_family_index];
}

/**
 * Helper used for debugging compiler issues by enabling/disabling LLVM for a
 * specific shader stage (developers only).
 */
static inline bool
radv_use_llvm_for_stage(const struct radv_physical_device *pdev, UNUSED mesa_shader_stage stage)
{
   return pdev->use_llvm;
}

bool radv_host_image_copy_enabled(const struct radv_physical_device *pdev);

bool radv_enable_rt(const struct radv_physical_device *pdev);

bool radv_emulate_rt(const struct radv_physical_device *pdev);

bool radv_use_bvh8(const struct radv_physical_device *pdev);

bool radv_is_dcc_disabled(const struct radv_physical_device *pdev);

bool radv_are_dcc_stores_disabled(const struct radv_physical_device *pdev);

bool radv_are_dcc_mips_disabled(const struct radv_physical_device *pdev);

uint32_t radv_find_memory_index(const struct radv_physical_device *pdev, VkMemoryPropertyFlags flags);

VkResult create_null_physical_device(struct vk_instance *vk_instance);

VkResult create_drm_physical_device(struct vk_instance *vk_instance, struct _drmDevice *device,
                                    struct vk_physical_device **out);

void radv_physical_device_destroy(struct vk_physical_device *vk_pdev);

bool radv_transfer_queue_enabled(const struct radv_physical_device *pdev);

bool radv_compute_queue_enabled(const struct radv_physical_device *pdev);

bool radv_spm_trace_enabled(const struct radv_physical_device *pdev);

static inline uint32_t
radv_get_sampled_image_desc_size(const struct radv_physical_device *pdev)
{
   /* Main descriptor + FMASK desccriptor if needed. */
   return 32 + (pdev->use_fmask ? 32 : 0);
}

static inline uint32_t
radv_get_combined_image_sampler_desc_size(const struct radv_physical_device *pdev)
{
   const uint32_t image_desc_size = radv_get_sampled_image_desc_size(pdev);

   return align(image_desc_size + RADV_SAMPLER_DESC_SIZE, 32);
}

static inline uint32_t
radv_get_combined_image_sampler_offset(const struct radv_physical_device *pdev)
{
   return radv_get_combined_image_sampler_desc_size(pdev) - RADV_SAMPLER_DESC_SIZE;
}

#endif /* RADV_PHYSICAL_DEVICE_H */
